// This file combines several files together with modifications. Whenever
// possible, functions were made static. Also anything that was C was changed
// to C++. The files are:
//
// common.h
// taputil.h
// adouble.h
// avector.h



//@@@@@@@ commom.h

/*---------------------------------------------------------------------------- 
 ADOL-C -- Automatic Differentiation by Overloading in C++
 File:     common.h
 Revision: $Id: common.h,v 1.2 2004/05/24 10:50:31 kowarz Exp $
 Contents: Common (global) ADOL-C header  

 Copyright (c) 2004
               Technical University Dresden
               Department of Mathematics
               Institute of Scientific Computing
  
 This file is part of ADOL-C. This software is provided under the terms of
 the Common Public License. Any use, reproduction, or distribution of the
 software constitutes recipient's acceptance of the terms of this license.
 See the accompanying copy of the Common Public License for more details.
----------------------------------------------------------------------------*/

#if !defined(ADOLC_COMMON_H)
#define ADOLC_COMMON_H 1

/*--------------------------------------------------------------------------*/
/* system dependend configuration */
#if HAVE_CONFIG_H
//#  include "config.h"
#endif

/*--------------------------------------------------------------------------*/
/* developer and user parameters */
#include "dvlparms.h"
#include "usrparms.h"

/*--------------------------------------------------------------------------*/
/* standard includes */
#include <stdlib.h>
#include <stdio.h>

/*--------------------------------------------------------------------------*/
/* malloc/calloc/ralloc replacments */
#if !defined(HAVE_MALLOC)
//#  include "malloc.h"
#  define malloc rpl_malloc
#  define calloc rpl_calloc
#endif
#if !defined(HAVE_REALLOC)
//#  include "malloc.h"
#  define realloc rpl_realloc
#endif

/*--------------------------------------------------------------------------*/
/* windows dll exports/imports */
#if defined(ADOLC_DLL)
#	define ADOLC_DLL_EXPORT __declspec(dllexport)
#else
#	define ADOLC_DLL_EXPORT
#endif

/*--------------------------------------------------------------------------*/
/* further helpful macros */
/* EGK make everything c++ */

/*#if defined(__cplusplus)
#  define BEGIN_C_DECLS extern "C" {
#  define END_C_DECLS   }
#else*/
#  define BEGIN_C_DECLS
#  define END_C_DECLS
/*#endif*/

#define MAXDEC(a,b) if ((a) < (b)) (a) = (b)
#define MINDEC(a,b) if ((a) > (b)) (a) = (b)

#define MAX_ADOLC(a,b) ( (a)<(b)? (b):(a) )
#define MIN_ADOLC(a,b) ( (a)>(b)? (b):(a) )

/*--------------------------------------------------------------------------*/
#endif


//@@@@@@@ malloc.h


/*---------------------------------------------------------------------------- 
 ADOL-C -- Automatic Differentiation by Overloading in C++
 File:     malloc.h
 Revision: $Id$
 Contents: malloc replacements for not gnu compatible malloc system functions

 Copyright (c) 2005
               Technical University Dresden
               Department of Mathematics
               Institute of Scientific Computing
  
 This file is part of ADOL-C. This software is provided under the terms of
 the Common Public License. Any use, reproduction, or distribution of the
 software constitutes recipient's acceptance of the terms of this license.
 See the accompanying copy of the Common Public License for more details.

History:
         20050617 kowarz: initial version

----------------------------------------------------------------------------*/

#if !defined(ADOLC_MALLOC_H)
#  define ADOLC_MALLOC_H 1

#if defined(HAVE_CONFIG_H)
//#  include "config.h"
#endif

#include <stddef.h>

#if defined(ADOLC_DLL)
#	define ADOLC_DLL_EXPORT __declspec(dllexport)
#else
#	define ADOLC_DLL_EXPORT
#endif

// EGK make pure c++
/*
#if defined(__cplusplus)
extern "C" {
#endif
*/

void *rpl_malloc(size_t);
void *rpl_calloc(size_t, size_t);
void *rpl_realloc(void *, size_t);

/*
#if defined(__cplusplus)
}
#endif
*/

#endif /* ADOLC_MALLOC_H */



//@@@@@@@  taputil.h

/*---------------------------------------------------------------------------- 
 ADOL-C -- Automatic Differentiation by Overloading in C++
 File:     taputil.h
 Revision: $Id: taputil.h,v 1.2 2004/05/24 10:50:41 kowarz Exp $
 Contents: Initialization & stopage of the taping process, as well as
           statistics gathering functions.

 Copyright (c) 2004
               Technical University Dresden
               Department of Mathematics
               Institute of Scientific Computing
  
 This file is part of ADOL-C. This software is provided under the terms of
 the Common Public License. Any use, reproduction, or distribution of the
 software constitutes recipient's acceptance of the terms of this license.
 See the accompanying copy of the Common Public License for more details.

 History:
          20030304 andrea: new variable for value stack name
          20030306 olvo:   extracted from taputil.h of ADOL-C 1.8.7
          19991122 olvo:   new op_codes eq_plus_prod eq_min_prod
                           for  y += x1 * x2
                           and  y -= x1 * x2
                           --> new: upd_resloc_inc_prod(..)  
          19990713 olvo:   trace_on/off: default values for arguments 
          19981130 olvo:   newly created by unification of taputil?.h
                           and all tape stuff

 History of taputil1.h:
          19980914 olvo:   adolcIDSize 5 (check size of locints ..)
          19980825 olvo:   #defines instead of const (C-Code!)
          19980820 olvo:   Version check
          19980723 olvo:   taputil3.* moved here
          19980713 olvo:   (1) no write_... routines anymore!
                           (2) statistic stuff kept here only
          19980709 olvo:   void write_pos_sign_a(..)
                           void write_neg_sign_a(..)
          19980708 olvo:   void write_upd(..)
          19980707 olvo:   void write_dot_av_av(..)
          19980706 olvo:   void write_incr_decr_a(..)
          19980623 olvo:   new operation code: take_stock_op

 History of taputil2.h:
          19980517 olvo:   griewank's idea:
                           int upd_resloc(locint, locint);
                     
----------------------------------------------------------------------------*/

#if !defined(ADOLC_TAPUTIL_H)
#define ADOLC_TAPUTIL_H 1

//#include "common.h"

BEGIN_C_DECLS

/****************************************************************************/
/*                                                       TRACING ON/OFF (C) */
ADOLC_DLL_EXPORT void start_trace(short,int);
ADOLC_DLL_EXPORT void stop_trace(int,int);

/****************************************************************************/
/*                                                          TAPE STATISTICS */
ADOLC_DLL_EXPORT void tapestats(short,int *);

END_C_DECLS

/****************************************************************************/
/****************************************************************************/
/*                                                       Now the C++ THINGS */
#if defined(__cplusplus)

/****************************************************************************/
/*                                                     TRACING ON/OFF (C++) */

ADOLC_DLL_EXPORT void trace_on( short, int = 0 ); 
ADOLC_DLL_EXPORT void trace_off( int = 0 );

#endif

/****************************************************************************/
#endif


//@@@@@@@  adouble.h


/*---------------------------------------------------------------------------- 
 ADOL-C -- Automatic Differentiation by Overloading in C++
 File:     adouble.h
 Revision: $Id: adouble.h,v 1.3 2004/11/10 14:25:24 kowarz Exp $
 Contents: adouble.h contains the basis for the class of adouble
           included here are all the possible functions defined on
           the adouble class.  Notice that, as opposed to ealier versions,
           both the class adub and the class adouble are derived from a base
           class (badouble).  See below for further explanation.

 Copyright (c) 2004
               Technical University Dresden
               Department of Mathematics
               Institute of Scientific Computing
  
 This file is part of ADOL-C. This software is provided under the terms of
 the Common Public License. Any use, reproduction, or distribution of the
 software constitutes recipient's acceptance of the terms of this license.
 See the accompanying copy of the Common Public License for more details.

 History:
          20041110 kowarz: tapeless (scalar/vector) forward version added
          20040423 kowarz: adapted to configure - make - make install
          20000107 olvo:   iostream.h instaed of stream.h  
          19991210 olvo:   checking the changes
          19991122 olvo:   new op_codes eq_plus_prod eq_min_prod
                           for  y += x1 * x2
                           and  y -= x1 * x2  
          19981201 olvo:   last check: 
                           - taputil things changed, includes 
          19980820 olvo:   new comparison strategy & some inlines
          19980709 olvo:   modified sign operators

----------------------------------------------------------------------------*/

/****************************************************************************/
/*
  NOTICE that the purpose of the class adub is merely to avoid the 
  generation and recording of an extra return adouble for each elementary 
  operation and function call. The same result can be achieved much
  more elegantly with GNUs named return variables, which would also 
  achieve the desired last in first out pattern for adouble construction 
  and destruction.
*/
/****************************************************************************/

#if !defined(ADOLC_ADOUBLE_H)
#define ADOLC_ADOUBLE_H 1

//#define BOUNDS_CHECK

/****************************************************************************/
/*                                                         THIS FILE IS C++ */
#ifdef __cplusplus

#include <cstdio>
#include <cstdlib>
#include <iostream>
#include <cmath>
using std::cout;
using std::cin;
using std::cerr;
using std::ostream;
using std::istream;

//#include "common.h"
//#include "taputil.h"

/* NOTICE: There are automatic includes at the end of this file! */

#undef TAPELESS
#undef SAFE
#if defined(ADOLC_TAPELESS)
#  define TAPELESS
#  undef SAFE
#endif

#if defined(SAFE_ADOLC_TAPELESS)
#  define TAPELESS
#  define SAFE
#endif

#if !defined(TAPELESS)

extern double* store; //added by EGK
extern int trace_flag; //added by EGK
extern locint maxloc; //added by EGK
extern locint current_top; //added by EGK
extern locint location_cnt; //added by EGK
extern locint maxtop; //added by EGK
extern locint maxtop2; //added by EGK
extern locint dealloc; //added by EGK
extern locint deminloc; //added by EGK
extern int currently_in_initial_timestep;

extern bool check_lifo;

inline void free_loc( locint old_loc )  //added by EGK
{ ++dealloc;
  if (old_loc < deminloc)
    deminloc = old_loc;

  // EGK
  /*
  if (check_lifo && dealloc+deminloc != current_top)
  {
     std::cout << "Error: All adouble variables must be allocated and "
	"deallocated in a last in first out basis. You appear to "
	"be violating this LIFO rule." << std::endl;

  std::cout << "old_loc: " << old_loc << "\n";
  std::cout << "dealloc: " << dealloc << "\n";
  std::cout << "deminloc: " << deminloc << "\n";
  std::cout << "current_top: " << current_top << "\n";
     abort();
 
  }
  */

}
/* Return the next free location in "adouble" memory */
//locint next_loc();
inline locint next_loc_fast() // added by EGK. removed some checks. Otherwise
			      // the same as next_loc
{ /* First deallocate dead adoubles if they form a contiguous tail: */

#ifdef overwrite
  if (dealloc && dealloc+deminloc == current_top)    
  { /* olvo 980518 deleted write_death (see version griewank) */
    // if (trace_flag) 
    //   write_death(deminloc, current_top - 1);
    current_top = deminloc ;
    dealloc     = 0; 
    deminloc    = maxloc;
//  std::cout << current_top;
  }

  //EGK begin
  else if (dealloc)// && dealloc+deminloc != current_top)
  {
     std::cout << "Error: All adouble variables must be allocated and "
	"deallocated in a last in first out basis. You appear to "
	"be violating this LIFO rule." << std::endl;
     std::cout << "dealloc: " << dealloc << "\n";
     std::cout << "deminloc: " << deminloc << "\n";
     std::cout << "current_top: " << current_top << "\n";
     abort();
  }
  //EGK end
#endif
  if (current_top == location_cnt)
  {
    ++location_cnt;
    /* Begin EGK */
    if(currently_in_initial_timestep == 0)
    {
       printf("Fatal error: attempt to add more adoubles "
	      "after initial timestep");
       abort();
     }
    /* End EGK */
  }
  if (location_cnt > maxtop) 
  { maxtop2 = ++maxtop*2 > maxloc ? maxloc : 2*maxtop;
    { maxtop = maxtop2;
      if (maxtop == 2)
      { store = (double *)malloc(maxtop*sizeof(double));
	deminloc = maxloc;
      }
      else
        store = (double *)realloc((char *)store,maxtop*sizeof(double));
      if (store == 0) 
      { fprintf(DIAG_OUT,"\nADOL-C error: in next_loc_fast\n");abort();
      }
    }
  }

  return current_top++;
}


/****************************************************************************/
/*                                             FORWARD DECLARATIONS (TAPES) */

/*--------------------------------------------------------------------------*/
class adouble;
class adub;
class badouble;
class badoublev;
class adoublev;
class adubv;
class along;
/* class doublev;  that's history */

/*--------------------------------------------------------------------------*/
/* // EGK: made static
void ADOLC_DLL_EXPORT condassign( double &res, const double &cond, 
                 const double &arg1, const double &arg2 );
void ADOLC_DLL_EXPORT condassign( double &res, const double &cond,
                 const double &arg );

double ADOLC_DLL_EXPORT fmin( const double &x, const double &y );
double ADOLC_DLL_EXPORT fmax( const double &x, const double &y );
*/


/****************************************************************************/
/*                                                           CLASS BADOUBLE */

/* 
   The class badouble contains the basic definitions for 
   the arithmetic operations, comparisons, etc. 
   This is a basic class from which the adub and adouble are 
   derived.  Notice that the constructors/destructors for 
   the class badouble are of the trivial variety.  This is the
   main difference among badoubles, adubs, and adoubles.
*/
class ADOLC_DLL_EXPORT badouble{
  friend ADOLC_DLL_EXPORT class badoublev;
 protected:
  locint location;
  badouble( void ){};
  // must be public when using gcc >= 3.4 ( problems with value() )
  // (see GCC 3.4 Release Series - Changes, New Features, and Fixes)
  //
  // badouble( const badouble& a ) {location = a.location;};
  badouble( locint lo ) {location = lo;};

 public:
/*--------------------------------------------------------------------------*/
  badouble( const badouble& a ) {location = a.location;};           /* ctor */
  
  inline locint loc( void ) const;                         /* Helpful stuff */

  /*------------------------------------------------------------------------*/
  badouble& operator >>= ( double& );                        /* Assignments */
  badouble& operator <<= ( double );
  inline badouble& operator = ( double coval); //made inline by EGK
  inline badouble& operator = ( const badouble& ); //made inline by EGK
  inline badouble& operator = ( const adub& ); //made inline by EGK
  double value() const {  return store[location]; } // made inline by EGK
  double& ref() {  return store[location]; } // added by EGK (dangerous!)
  const double& ref() const {  return store[location]; } // added by EGK (dangerous!)
   
  /* badouble& operator = ( const adouble& ); 
     !!! olvo 991210: was the same as badouble-assignment */

   badouble& operator_equal ( double ); //added by EGK
   badouble& operator_equal ( const badouble& ); //added by EGK
   badouble& operator_equal ( const adub& ); //added by EGK

   
/*--------------------------------------------------------------------------*/
  friend ADOLC_DLL_EXPORT std::ostream& operator << ( std::ostream&, const badouble& );  /* IO friends */
  friend ADOLC_DLL_EXPORT std::istream& operator >> ( std::istream&, const badouble& );

  /*------------------------------------------------------------------------*/
                                                  /* Operation + Assignment */
  inline badouble& operator += ( double );  //made inline by EGK
  inline badouble& operator += ( const badouble& );  //made inline by EGK
  inline badouble& operator -= ( double y );  //made inline by EGK
  inline badouble& operator -= ( const badouble& );  //made inline by EGK
  inline badouble& operator *= ( double );  //made inline by EGK
  inline badouble& operator *= ( const badouble& );  //made inline by EGK
  inline badouble& operator /= ( double );  //made inline by EGK
  inline badouble& operator /= ( const badouble& );  //made inline by EGK
  /* olvo 991122 n2l: new special op_codes */
  inline badouble& operator += ( const adub& );  //made inline by EGK
  inline badouble& operator -= ( const adub& );  //made inline by EGK

   
    badouble& operator_plus_equal ( double );  // added by EGK
    badouble& operator_plus_equal ( const badouble& );  // added by EGK
    badouble& operator_minus_equal ( double y );  // added by EGK
    badouble& operator_minus_equal ( const badouble& );  // added by EGK
    badouble& operator_mult_equal ( double );  // added by EGK
    badouble& operator_mult_equal ( const badouble& );  //added by EGK
    badouble& operator_plus_equal ( const adub& a );  //added by EGK
    badouble& operator_minus_equal ( const adub& a );  //added by EGK

    friend ADOLC_DLL_EXPORT adub operator_mult ( const badouble&, const badouble& ); //added by EGK
//    friend ADOLC_DLL_EXPORT badouble& operator_mult ( const badouble&, const badouble& ); //added by EGK
    friend ADOLC_DLL_EXPORT adub operator_mult ( double, const badouble& );  //added by EGK
//    friend ADOLC_DLL_EXPORT badouble& operator_mult ( double, const badouble& );  //added by EGK
    friend ADOLC_DLL_EXPORT adub operator_plus ( const badouble& x, const badouble& y );  //added by EGK
    friend ADOLC_DLL_EXPORT adub operator_plus ( double coval, const badouble& y ); //added by EGK
    friend ADOLC_DLL_EXPORT adub operator_plus ( const badouble& y, double coval ); //added by EGK
    friend ADOLC_DLL_EXPORT adub operator_minus ( const badouble& x, const badouble& y ); //added by EGK
    friend ADOLC_DLL_EXPORT adub operator_minus ( double coval, const badouble& y ); //added by EGK
    friend ADOLC_DLL_EXPORT adub operator_divide ( const badouble& x, const badouble& y ); //added by EGK
    friend ADOLC_DLL_EXPORT adub operator_divide ( double coval, const badouble& y ); //added by EGK
   
/*--------------------------------------------------------------------------*/
                                                    /* Comparison (friends) */
  inline friend int operator != ( const badouble&, const badouble& );
  inline friend int operator != ( double, const badouble& );
         friend ADOLC_DLL_EXPORT int operator != ( const badouble&, double );
  inline friend int operator == ( const badouble&, const badouble& );
  inline friend int operator == ( double, const badouble& );
         friend ADOLC_DLL_EXPORT int operator == ( const badouble&, double );
  inline friend int operator <= ( const badouble&, const badouble& );
  inline friend int operator <= ( double, const badouble& );
         friend ADOLC_DLL_EXPORT int operator <= ( const badouble&, double );
  inline friend int operator >= ( const badouble&, const badouble& );
  inline friend int operator >= ( double, const badouble& );
         friend ADOLC_DLL_EXPORT int operator >= ( const badouble&, double );
  inline friend int operator >  ( const badouble&, const badouble& );
  inline friend int operator >  ( double, const badouble& );
         friend ADOLC_DLL_EXPORT int operator >  ( const badouble&, double );
  inline friend int operator <  ( const badouble&, const badouble& );
  inline friend int operator <  ( double, const badouble& );
         friend ADOLC_DLL_EXPORT int operator <  ( const badouble&, double );


/*--------------------------------------------------------------------------*/
                                                /* sign operators (friends) */
         friend ADOLC_DLL_EXPORT adub operator + ( const badouble& x ); 
         friend ADOLC_DLL_EXPORT adub operator - ( const badouble& x ); 

/*--------------------------------------------------------------------------*/
                                              /* binary operators (friends) */
   inline friend ADOLC_DLL_EXPORT adub operator + ( const badouble&, const badouble& );  //made inline by EGK
   inline friend ADOLC_DLL_EXPORT adub operator + ( double, const badouble& ); //made inline by EGK
   inline friend ADOLC_DLL_EXPORT adub operator + ( const badouble&, double ); //made inline by EGK
   inline friend ADOLC_DLL_EXPORT adub operator - ( const badouble&, const badouble& ); //made inline by EGK
   inline friend adub operator - ( const badouble&, double ); 
   inline friend ADOLC_DLL_EXPORT adub operator - ( double, const badouble& ); //made inline by EGK
      inline friend ADOLC_DLL_EXPORT adub operator * ( const badouble&, const badouble& ); //made inline by EGK
//      inline friend ADOLC_DLL_EXPORT badouble& operator * ( const badouble&, const badouble& ); // made inline by EGK
    inline friend ADOLC_DLL_EXPORT adub operator * ( double, const badouble& );  //made inline by EGK
//    inline friend ADOLC_DLL_EXPORT badouble& operator * ( double, const badouble& );  //made inline by EGK
  inline friend adub operator * ( const badouble&, double );
//  inline friend badouble& operator * ( const badouble&, double );
  inline friend adub operator / ( const badouble&, double );
//  inline friend badouble& operator / ( const badouble&, double );
  inline friend ADOLC_DLL_EXPORT adub operator / ( const badouble&, const badouble& ); // made inline by EGK
  inline friend ADOLC_DLL_EXPORT adub operator / ( double, const badouble& ); // made inline by EGK

/*--------------------------------------------------------------------------*/
                                               /* unary operators (friends) */
         friend ADOLC_DLL_EXPORT adub exp  ( const badouble& ); 
         friend ADOLC_DLL_EXPORT adub log  ( const badouble& ); 
         friend ADOLC_DLL_EXPORT adub sqrt ( const badouble& );
         friend ADOLC_DLL_EXPORT adub sin  ( const badouble& ); 
         friend ADOLC_DLL_EXPORT adub cos  ( const badouble& );
         friend ADOLC_DLL_EXPORT adub tan  ( const badouble& );
         friend ADOLC_DLL_EXPORT adub asin ( const badouble& );
         friend ADOLC_DLL_EXPORT adub acos ( const badouble& );
         friend ADOLC_DLL_EXPORT adub atan ( const badouble& ); 

/*--------------------------------------------------------------------------*/
                                             /* special operators (friends) */
         friend ADOLC_DLL_EXPORT adouble atan2 ( const badouble&, const badouble& ); 
         /* no internal use of condassign: */
         friend ADOLC_DLL_EXPORT adub    pow   ( const badouble&, double );
         /* uses condassign internally */
         friend ADOLC_DLL_EXPORT adouble pow   ( const badouble&, const badouble& );
         friend ADOLC_DLL_EXPORT adouble pow   ( double, const badouble& );
         friend ADOLC_DLL_EXPORT adub    log10 ( const badouble& );
         /* User defined version of logarithm to test extend_quad macro */
         friend ADOLC_DLL_EXPORT adouble myquad( const badouble& );

/*--------------------------------------------------------------------------*/
        /* Additional ANSI C standard Math functions Added by DWJ on 8/6/90 */
         friend ADOLC_DLL_EXPORT adub sinh  ( const badouble& );
         friend ADOLC_DLL_EXPORT adub cosh  ( const badouble& );
         friend ADOLC_DLL_EXPORT adub tanh  ( const badouble& );
         friend ADOLC_DLL_EXPORT adub asinh ( const badouble& );
         friend ADOLC_DLL_EXPORT adub acosh ( const badouble& );
         friend ADOLC_DLL_EXPORT adub atanh ( const badouble& );

         friend ADOLC_DLL_EXPORT adub fabs  ( const badouble& );
         friend ADOLC_DLL_EXPORT adub ceil  ( const badouble& );
         friend ADOLC_DLL_EXPORT adub floor ( const badouble& );

         friend ADOLC_DLL_EXPORT adub fmax ( const badouble&, const badouble& );
         friend ADOLC_DLL_EXPORT adub fmax ( double, const badouble& );
         friend ADOLC_DLL_EXPORT adub fmax ( const badouble&, double );
         friend ADOLC_DLL_EXPORT adub fmin ( const badouble&, const badouble& );
         friend ADOLC_DLL_EXPORT adub fmin ( double, const badouble& );
         friend ADOLC_DLL_EXPORT adub fmin ( const badouble&, double );

         friend ADOLC_DLL_EXPORT adub ldexp ( const badouble&, int );
         friend ADOLC_DLL_EXPORT adub frexp ( const badouble&, int* );
         friend ADOLC_DLL_EXPORT adub erf   ( const badouble& );

/*--------------------------------------------------------------------------*/
                                                            /* Conditionals */
         friend ADOLC_DLL_EXPORT void condassign( adouble &res, const adouble &cond,
                          const adouble &arg1, const adouble &arg2 );
         friend ADOLC_DLL_EXPORT void condassign( adouble &res, const adouble &cond,
                                               const adouble &arg );
         friend ADOLC_DLL_EXPORT void condassign( along &res, const adouble &cond,
                        const adouble &arg1, const adouble &arg2 );
         friend ADOLC_DLL_EXPORT void condassign( along &res, const adouble &cond,
                                             const adouble &arg );
};




/****************************************************************************/
/*                                                               CLASS ADUB */

/* 
   The class Adub
   ---- Basically used as a temporary result.  The address for an
        adub is usually generated within an operation.  That address
        is "freed" when the adub goes out of scope (at destruction time).
   ---- operates just like a badouble, but it has a destructor defined for it.
*/

class ADOLC_DLL_EXPORT adub:public badouble{
  friend ADOLC_DLL_EXPORT class adouble;
/* added Sep/01/96 */
  friend ADOLC_DLL_EXPORT class asub;
  friend ADOLC_DLL_EXPORT class along;
 protected:
  adub( locint lo ):badouble(lo){};
  adub( void ):badouble(0)
  { fprintf(DIAG_OUT,"ADOL-C error: illegal default construction of adub"
                     " variable\n");
    exit(-2);
  };
  adub( double ):badouble(0)
  { fprintf(DIAG_OUT,"ADOL-C error: illegal  construction of adub variable"
                     " from double\n");
    exit(-2);
  };

 public:


    friend ADOLC_DLL_EXPORT adub operator_mult ( const badouble&, const badouble& ); //added by EGK
//    friend ADOLC_DLL_EXPORT badouble& operator_mult ( const badouble&, const badouble& ); //added by EGK
    friend ADOLC_DLL_EXPORT adub operator_mult ( double, const badouble& );  //added by EGK
//    friend ADOLC_DLL_EXPORT badouble& operator_mult ( double, const badouble& );  //added by EGK
    friend ADOLC_DLL_EXPORT adub operator_plus ( const badouble& x, const badouble& y );  //added by EGK
    friend ADOLC_DLL_EXPORT adub operator_plus ( double coval, const badouble& y ); //added by EGK
    friend ADOLC_DLL_EXPORT adub operator_plus ( const badouble& y, double coval ); //added by EGK
    friend ADOLC_DLL_EXPORT adub operator_minus ( const badouble& x, const badouble& y ); //added by EGK
    friend ADOLC_DLL_EXPORT adub operator_minus ( double coval, const badouble& y ); //added by EGK
    friend ADOLC_DLL_EXPORT adub operator_divide ( const badouble& x, const badouble& y ); //added by EGK
    friend ADOLC_DLL_EXPORT adub operator_divide ( double coval, const badouble& y ); //added by EGK

/*--------------------------------------------------------------------------*/
                                                /* sign operators (friends) */
         friend ADOLC_DLL_EXPORT adub operator + ( const badouble& x ); 
         friend ADOLC_DLL_EXPORT adub operator - ( const badouble& x ); 

/*--------------------------------------------------------------------------*/
                                              /* binary operators (friends) */
   inline friend ADOLC_DLL_EXPORT adub operator + ( const badouble&, const badouble& );   //made inline by EGK
   inline friend ADOLC_DLL_EXPORT adub operator + ( double, const badouble& ); //made inline by EGK
   inline friend ADOLC_DLL_EXPORT adub operator + ( const badouble&, double ); //made inline by EGK
   inline friend ADOLC_DLL_EXPORT adub operator - ( const badouble&, const badouble& ); //made inline by EGK
   inline friend adub operator - ( const badouble&, double ); 
   inline friend ADOLC_DLL_EXPORT adub operator - ( double, const badouble& ); //made inline by EGK
      inline friend ADOLC_DLL_EXPORT adub operator * ( const badouble&, const badouble& ); // made inline by EGK
//      inline friend ADOLC_DLL_EXPORT badouble& operator * ( const badouble&, const badouble& ); // made inline by EGK
      inline friend ADOLC_DLL_EXPORT adub operator * ( double, const badouble& ); // made inline by EGK
//      inline friend ADOLC_DLL_EXPORT badouble& operator * ( double, const badouble& ); // made inline by EGK
  inline friend adub operator * ( const badouble&, double );
//  inline friend badouble& operator * ( const badouble&, double );
  inline friend adub operator / ( const badouble&, double );
//  inline friend badouble& operator / ( const badouble&, double );
  inline friend ADOLC_DLL_EXPORT adub operator / ( const badouble&, const badouble& ); // made inline by EGK
  inline friend ADOLC_DLL_EXPORT adub operator / ( double, const badouble& ); // made inline by EGK

/*--------------------------------------------------------------------------*/
                                               /* unary operators (friends) */
         friend ADOLC_DLL_EXPORT adub exp  ( const badouble& ); 
         friend ADOLC_DLL_EXPORT adub log  ( const badouble& ); 
         friend ADOLC_DLL_EXPORT adub sqrt ( const badouble& );
         friend ADOLC_DLL_EXPORT adub sin  ( const badouble& ); 
         friend ADOLC_DLL_EXPORT adub cos  ( const badouble& );
         friend ADOLC_DLL_EXPORT adub tan  ( const badouble& );
         friend ADOLC_DLL_EXPORT adub asin ( const badouble& );
         friend ADOLC_DLL_EXPORT adub acos ( const badouble& );
         friend ADOLC_DLL_EXPORT adub atan ( const badouble& ); 

/*--------------------------------------------------------------------------*/
                                             /* special operators (friends) */
         /* no internal use of condassign: */
         friend ADOLC_DLL_EXPORT adub    pow   ( const badouble&, double );
         friend ADOLC_DLL_EXPORT adub    log10 ( const badouble& );

/*--------------------------------------------------------------------------*/
        /* Additional ANSI C standard Math functions Added by DWJ on 8/6/90 */
         friend ADOLC_DLL_EXPORT adub sinh  ( const badouble& );
         friend ADOLC_DLL_EXPORT adub cosh  ( const badouble& );
         friend ADOLC_DLL_EXPORT adub tanh  ( const badouble& );
         friend ADOLC_DLL_EXPORT adub asinh ( const badouble& );
         friend ADOLC_DLL_EXPORT adub acosh ( const badouble& );
         friend ADOLC_DLL_EXPORT adub atanh ( const badouble& );

         friend ADOLC_DLL_EXPORT adub fabs  ( const badouble& );
         friend ADOLC_DLL_EXPORT adub ceil  ( const badouble& );
         friend ADOLC_DLL_EXPORT adub floor ( const badouble& );

         friend ADOLC_DLL_EXPORT adub fmax ( const badouble&, const badouble& );
         friend ADOLC_DLL_EXPORT adub fmax ( double, const badouble& );
         friend ADOLC_DLL_EXPORT adub fmax ( const badouble&, double );
         friend ADOLC_DLL_EXPORT adub fmin ( const badouble&, const badouble& );
         friend ADOLC_DLL_EXPORT adub fmin ( double, const badouble& );
         friend ADOLC_DLL_EXPORT adub fmin ( const badouble&, double );

         friend ADOLC_DLL_EXPORT adub ldexp ( const badouble&, int );
         friend ADOLC_DLL_EXPORT adub frexp ( const badouble&, int* );
         friend ADOLC_DLL_EXPORT adub erf   ( const badouble& );

/*--------------------------------------------------------------------------*/
	                                     /* vector operations (friends) */
         friend ADOLC_DLL_EXPORT adub operator*( const badoublev&, double* );
         friend ADOLC_DLL_EXPORT adub operator*( double*, const badoublev& );
         friend ADOLC_DLL_EXPORT adub operator*( const badoublev&, const badoublev& );

  //added by EGK
  ADOLC_DLL_EXPORT friend adub operator_mult ( const badoublev &arg1, const badoublev &arg2 );

#ifdef overwrite
      ~adub(){ free_loc(location);} // made inline by EGK
#endif
};


/****************************************************************************/
/*                                                            CLASS ADOUBLE */
/*
  The class adouble.
  ---Derived from badouble.  Contains the standard constructors/destructors.
  ---At construction, it is given a new address, and at destruction, that
     address is freed.
*/
class ADOLC_DLL_EXPORT adouble:public badouble{
  friend ADOLC_DLL_EXPORT class along;
 public:
  adouble( const adub& );
  adouble( const along& );
  adouble( const adouble& );
  adouble( void );
  adouble( double );
  /* adub prevents postfix operators to occur on the left 
     side of an assignment which would not work  */
  adub operator++( int );
  adub operator--( int );
  badouble& operator++( void );
  badouble& operator--( void );
/*   inline double value(); */
#ifdef overwrite
  ~adouble();
#endif

  adouble& operator = ( double coval ) // made inline by EGK
  { (*this).badouble::operator=(coval);
     return (*this); }
   
  adouble& operator = ( const badouble& x) // made inline by EGK
  { (*this).badouble::operator=(x);
     return (*this); }

   /* adouble& operator = ( const adouble& );
     !!! olvo 991210 was the same as badouble-assignment */ 
  adouble& operator = ( const adub& a) // made inline by EGK
  { (*this).badouble::operator=(a);
     return (*this); }

};

/****************************************************************************/
/*                                                               CLASS ASUB */
class ADOLC_DLL_EXPORT asub:public badouble
{ locint base,
         offset;
 public:
  asub( locint start, locint index );
#ifdef overwrite
  ~asub();
#endif
  asub& operator <<= ( double );
  asub& operator =   ( double );
  /* asub& operator =   ( const adub& );
     !!! olvo 991210 is the same as normal assignment */
  asub& operator =   ( const badouble& );
  /* added Sep/01/96 */
  /* olvo 991210 seems to be a provisional version */
  asub& operator =   ( const asub& );
  asub& operator +=  ( double );
  asub& operator +=  ( const badouble& );
  asub& operator -=  ( double x );
  asub& operator -=  ( const badouble& );
  asub& operator *=  ( double x );
  asub& operator *=  ( const badouble& );
  asub& operator /=  ( double x );
  asub& operator /=  ( const badouble& );
  /* adub prevents postfix operators to occur on the left 
     side of an assignment which would not work  */
  adub operator++( int );
  adub operator--( int );
  asub& operator++( void );
  asub& operator--( void );
};


/****************************************************************************/
/*                                                              CLASS ALONG */
/* The class along was originally designed for the sole purpose of 
   allowing subscripting operations with computable ON THE TAPE.
   The current definition refers to badouble, i.e. it is a 
   floating point class. The current constuction allows to 
   use alongs in the same way as adoubles. Especially adubs can 
   can be used as subscripts, i.e. the results of "active computations".
   This useful because people want to compute the indices on the tape.
   Notice, that along does NOT perform integer arithmetic. 
   This is the major disadvantage of the current along. */
class ADOLC_DLL_EXPORT along:public badouble
{ friend ADOLC_DLL_EXPORT class adouble;
 public:
  along( const adub& );
  along( const along& );
  along( void );
  along( int );
#ifdef overwrite
  ~along();
#endif
  along& operator = ( int );
  /* along& operator = ( const adouble& );
     !!! olvo 991210 is the same as badouble-assignment */ 
  along& operator = ( const badouble& );
  along& operator = ( const along& );
  along& operator = ( const adub& );
  /* adub prevents postfix operators to occur on the left
     side of an assignment which would not work  */
  adub operator++( int );
  adub operator--( int );
  along&  operator++( void );
  along&  operator--( void );
};



/****************************************************************************/
/*                                                       INLINE DEFINITIONS */



/*--------------------------------------------------------------------------*/
inline locint badouble::loc( void ) const
{ return location;
} 

inline badouble& badouble::operator = ( double coval)
{ if (trace_flag) 
  {  
     return operator_equal(coval);
  }

 store[location] = coval;
 return *this;

}
  
inline badouble& badouble::operator = ( const badouble& x)
{ if (trace_flag) 
  {  
     return operator_equal(x);
  }

//locint x_loc = x.loc();
//if (location!=x_loc)  
 store[location]=store[x.loc()];
 return *this;

}

inline badouble& badouble::operator = ( const adub& a )
{ 
   if(trace_flag)
   {
     return operator_equal(a);
   }

   //locint a_loc = a.loc();
   store[location] = store[a.loc()];
   return *this;

}



/*--------------------------------------------------------------------------*/
/* Adding a floating point to an adouble */
inline badouble& badouble::operator += ( double coval ) 
{ if (trace_flag) 
  { 
     return operator_plus_equal(coval);
  } 

     store[location] += coval;
     return *this; 

}


/*--------------------------------------------------------------------------*/
/* Subtracting a floating point from an adouble */
inline badouble& badouble::operator -= ( double coval ) 
{ 
  if (trace_flag) 
  { 
     return operator_minus_equal(coval);
  } 

  store[location] -= coval;
  return *this; 

}

/*--------------------------------------------------------------------------*/
/* Add an adouble to another adouble */
inline badouble& badouble::operator += ( const badouble& y ) 
{ 
   if (trace_flag) 
  { 
     return operator_plus_equal(y);
  } 

     store[location] += store[y.loc()];
     return *this; 


}

/*--------------------------------------------------------------------------*/
/* Subtract an adouble from another adouble */
inline badouble& badouble::operator -= ( const badouble& y ) 
{ 
  if (trace_flag) 
  { 
     return operator_minus_equal(y);
  } 

     store[location] -= store[y.loc()];
     return *this; 

}

/*--------------------------------------------------------------------------*/
/* Multiply an adouble by a floating point */
inline badouble& badouble::operator *= ( double coval ) 
{ 
   if (trace_flag) 
  { 
     return operator_mult_equal(coval);
  } 

     store[location] *= coval;
     return *this; 

}

/*--------------------------------------------------------------------------*/
/* Multiply one adouble by another adouble*/
inline badouble& badouble::operator *= ( const badouble& y ) 
{
  if (trace_flag) 
  { 
     return operator_mult_equal(y);
  } 

     store[location] *= store[y.loc()];
     return *this; 

}

inline badouble& badouble::operator += ( const adub& a )
{   if (trace_flag) 
  { 
     return operator_plus_equal(a);
  } 

    store[location] += store[a.loc()];
     return *this; 

}

inline badouble& badouble::operator -= ( const adub& a )
{   if (trace_flag) 
  { 
     return operator_minus_equal(a);
  } 

    store[location] -= store[a.loc()];
     return *this; 

}

/*--------------------------------------------------------------------------*/
inline badouble& badouble::operator /= (double y) 
{ *this = *this/y;
  return *this;
}

/*--------------------------------------------------------------------------*/
inline badouble& badouble::operator /= (const badouble& y) 
{ *this = *this * (1.0/y);
  return *this;
}

/*--------------------------------------------------------------------------*/
                                                              /* Comparison */
inline int operator != ( const badouble& u, const badouble& v )
{ return (u-v != 0);
}

inline int operator != ( double coval, const badouble& v)
{ if (coval)
    return (-coval+v != 0);
  else
    return (v != 0);
}

inline int operator == ( const badouble& u, const badouble& v )
{ return (u-v == 0);
}

inline int operator == ( double coval, const badouble& v)
{ if (coval)
    return (-coval+v == 0);
  else
    return (v == 0);
}

inline int operator <= ( const badouble& u, const badouble& v )
{ return (u-v <= 0);
}

inline int operator <= ( double coval, const badouble& v )
{ if (coval)
    return (-coval+v >= 0);
  else
    return (v >= 0);
}

inline int operator >= ( const badouble& u, const badouble& v )
{ return (u-v >= 0);
}

inline int operator >= ( double coval, const badouble& v )
{ if (coval)
    return (-coval+v <= 0);
  else
    return (v <= 0);
}

inline int operator > ( const badouble& u, const badouble& v )
{ return (u-v > 0);
}

inline int operator > ( double coval, const badouble& v )
{ if (coval)
    return (-coval+v < 0);
  else
    return (v < 0);
}

inline int operator < ( const badouble& u, const badouble& v )
{ return (u-v < 0);
}

inline int operator < ( double coval, const badouble& v )
{ if (coval)
    return (-coval+v > 0);
  else
    return (v > 0);
}

/*--------------------------------------------------------------------------*/
/* Subtract a floating point from an adouble  */
inline adub operator - ( const badouble& x , double coval )
{ return (-coval) + x;
}

/*--------------------------------------------------------------------------*/
/* Multiply an adouble by a floating point */ 

inline adub operator * (const badouble& y, double coval)
{ 
   //EGK We replaced this with the actual function
   //return coval * y;

   if (trace_flag)
    { // old: write_pos_sign_a(locat,y.location);
       return operator_mult(coval,y);
    }   

   locint locat = next_loc_fast();

// EGK removed checks for 1 and -1 since uncommon
//  if ( coval == 1.0 )
//  { 
//    store[locat] = store[y.location];  
//  }
//  else
//    if ( coval == -1.0 )
//    {
//      store[locat] = -store[y.location];  
//    }
//    else
  {
      store[locat] = coval * store[y.location];
    }
  return locat;

}
/*
inline badouble& operator * (const badouble& x, double coval)
{ return coval * x;
}
*/
/*--------------------------------------------------------------------------*/
/* Divide an adouble by a floating point */

inline adub operator / (const badouble& x, double coval)
{ return (1.0/coval) * x;
}
/*
inline badouble& operator / (const badouble& x, double coval)
{ return (1.0/coval) * x;
}
*/

inline adub operator * ( const badouble& x, const badouble& y )
{ 
  if (trace_flag) 
  {
     return operator_mult(x,y);
  }
  
  locint locat = next_loc_fast();
  store[locat] = store[x.location] * store[y.location];
  return locat;
}

/*--------------------------------------------------------------------------*/
/* Multiply an adouble by a floating point */ 
/* olvo 980709 modified */

inline adub operator * ( double coval, const badouble& y )
{ 
   if (trace_flag)
    { // old: write_pos_sign_a(locat,y.location);
       return operator_mult(coval,y);
    }   

   locint locat = next_loc_fast();

// EGK removed checks for 1 and -1 since uncommon
//  if ( coval == 1.0 )
//  { 
//    store[locat] = store[y.location];  
//  }
//  else
//    if ( coval == -1.0 )
//    {
//      store[locat] = -store[y.location];  
//    }
//    else
  {
      store[locat] = coval * store[y.location];
    }
  return locat;
}

/*--------------------------------------------------------------------------*/
/* Adding two adoubles */
inline adub operator + ( const badouble& x, const badouble& y ) 
{ 
  if (trace_flag) 
  { // old: write_two_a_rec(plus_a_a,locat,x.location,y.location);
     return operator_plus(x,y);

  }

   locint locat = next_loc_fast();

  store[locat] = store[x.location] + store[y.location];
  return locat;
}

/*--------------------------------------------------------------------------*/
/* Adding a adouble and a floating point */
inline adub operator + ( double coval, const badouble& y ) 
{ 

  if (trace_flag) 
  { 
     return operator_plus(coval,y);
  }

  locint locat = next_loc_fast(); 

  /* olvo 980708 test coval to be zero */
  // EGK removed test for zero since uncommon
//  if (coval)
  { 
    store[locat] = coval + store[y.location];
  } 
//  else 
//  { 
//    store[locat] = store[y.location];
//  }

  return locat;
}

/*--------------------------------------------------------------------------*/
inline adub operator + ( const badouble& y, double coval) 
{ 

  if (trace_flag) 
  { 
     return operator_plus(y,coval);
  }

  locint locat = next_loc_fast(); 

  /* olvo 980708 test coval to be zero */
  // EGK removed test for zero since uncommon
//  if (coval)
  { 
    store[locat] = coval + store[y.location];
  } 
//  else 
//  { 
//    store[locat] = store[y.location];
//  }

  return locat;
}

/*--------------------------------------------------------------------------*/
/* Subtraction of two adoubles */
inline adub operator - ( const badouble& x, const badouble& y )
{ 

  if (trace_flag) 
  { 
     return operator_minus(x,y);
  }
  
  locint locat = next_loc_fast();

  store[locat] = store[x.location] - store[y.location];
  return locat;
}


/*--------------------------------------------------------------------------*/
/* Subtract an adouble from a floating point */
inline adub operator - ( double coval, const badouble& y )
{ 

  if (trace_flag) 
  { 
     return operator_minus(coval,y);
  }

  locint locat = next_loc_fast();

  /* olvo 980708 test coval to be zero */
  // EGK removed test for zero since uncommon
//  if (coval)
  { 
    store[locat] = coval - store[y.location];
  }
//  else
//  { 
//    store[locat] = -store[y.location];  
//  }

  return locat; 
}

/*--------------------------------------------------------------------------*/
/* Divide an adouble by another adouble */
inline adub operator / ( const badouble& x, const badouble& y )
{ 

  if (trace_flag) 
  { 
     return operator_divide(x,y);
  }


  locint locat = next_loc_fast();

  store[locat] = store[x.location] / store[y.location];
  return locat;
}

/*--------------------------------------------------------------------------*/
/* Division floating point - adouble */
inline adub operator / ( double coval, const badouble& y )
{ 
  if (trace_flag) 
  { 
     return operator_divide(coval,y);
  }

  locint locat = next_loc_fast();

  store[locat] = coval  / store[y.location];
  return locat;
}


/****************************************************************************/
/*                                                        AUTOMTIC INCLUDES */
//#include "avector.h" /* active vector classes */

/****************************************************************************/
/* tapeless implementation                                                  */
/****************************************************************************/
#else

// EGK: I removed the tapeless implementation since it's not needed here

#endif /* ADOLC_TAPELESS */

/****************************************************************************/
/*                                                                THAT'S ALL*/
#endif /* __cplusplus */
#endif /* ADOLC_ADOUBLE_H */



//@@@@@@@ avector.h

/*---------------------------------------------------------------------------- 
 ADOL-C -- Automatic Differentiation by Overloading in C++
 File:     avector.h
 Revision: $Id: avector.h,v 1.3 2004/11/10 14:25:24 kowarz Exp $
 Contents: Avector.h defines classes of vectors and matrices.
           badoublev  --> class of basic active vectors. 
           adubv      --> class of temporary active vectors.
                      (derived from badoublev.  Contains copy constructors, 
                      destructors.)
           adoublev   --> class of active vectors. (derived from badoublev,
                      contains the standard constructors and destructors.
                  
 Copyright (c) 2004
               Technical University Dresden
               Department of Mathematics
               Institute of Scientific Computing
  
 This file is part of ADOL-C. This software is provided under the terms of
 the Common Public License. Any use, reproduction, or distribution of the
 software constitutes recipient's acceptance of the terms of this license.
 See the accompanying copy of the Common Public License for more details.

 History:
          20041110 kowarz: ifdefs for tapeless forward added
          20040423 kowarz: adapted configure - make - make install
          19981130 olvo:   last check (includes ...)
	                        NOTICE: I think everything concerning vectors 
                                   has to be checked again in detail!
 
----------------------------------------------------------------------------*/

#if !defined(ADOLC_AVECTOR_H)
#define ADOLC_AVECTOR_H 1

#if !defined(ADOLC_TAPELESS)

//#include "common.h"

/****************************************************************************/
/*                                                         THIS FILE IS C++ */
#ifdef __cplusplus

/****************************************************************************/
/*                                                     FORWARD DECLARATIONS */
class badoublev;
class adoublev;  
class adubv;
/* class doublev; removed 1/95 */
class err_retu;
class asubv;


/****************************************************************************/
/*                                                          ANY ERROR CLASS */
class ADOLC_DLL_EXPORT err_retu
{ char* message;
 public:
  err_retu(char* x){printf("%s \n",x);};
};


/****************************************************************************/
/*                                            DECLARATION OF VECTOR CLASSES */

/* Passive vectors and matrices were REMOVED 1/95  */


/****************************************************************************/
/*                                                          CLASS BADOUBLEV */
class ADOLC_DLL_EXPORT badoublev
{
 protected:
  locint start_loc;  /* Starting location of vector in store */
  int size;          /* Size of the vector */
  badoublev(){};
  badoublev(int lo, int sz){start_loc = lo; size=sz;};
  badoublev(const badoublev& a){start_loc = a.start_loc; size=a.size;};
  
 public:

  /* Access functions */
  int sz() const {return size;}  /* Get the size of the vector */
  locint loc() const {return start_loc;}  /* Get the size of the vector */

  asub operator[](const along&) const;

/* excluded before 1/95 
  badoublev& operator >>= (doublev& );
  badoublev& operator <<= (doublev& );
  badoublev& operator >>= (double* );
  badoublev& operator <<= (double* );
*/

  // EGK: made inline by EGK, removed bounds checking
  badouble operator[](int i) const  /* Can access component like an array */
      {
#ifdef BOUNDS_CHECK
	 if (i<0 || i>=size)
	 { fprintf (DIAG_OUT,"ADOL-C error: adoublev index out of range.\n");
	 exit(-3);
	 }
#endif
	 return start_loc+i;}

  double& ref(int i) { return store[start_loc+i]; } // added by EGK (dangerous!)
  const double& ref(int i) const {return store[start_loc+i]; }//added by EGK(dangerous!)

  badoublev& operator+=(const badoublev&);
  badoublev& operator-=(const badoublev&);
  badoublev& operator*=(double);
  badoublev& operator/=(double);
/* removed 1/95
  badoublev& operator-=(const doublev&);
  badoublev& operator+=(const doublev&);
*/
/* removed Sep/01/96 
  badoublev& operator-=(double*);
  badoublev& operator+=(double*);
*/
  badoublev& operator*=(const badouble& );
  badoublev& operator/=(const badouble& );
  ADOLC_DLL_EXPORT friend adubv operator/(const badoublev &op1, const badouble &n);
  inline ADOLC_DLL_EXPORT friend adubv operator/(const badoublev &op1, double n);
/*  removed 1/95
  badoublev& operator= (const doublev&);
*/
  badoublev& operator= (const badoublev&);
  badoublev& operator= (const adubv &y);
  badoublev& operator= (const adoublev &y);

  //added by EGK
  ADOLC_DLL_EXPORT friend adub operator_mult ( const badoublev &arg1, const badoublev &arg2 );


  ADOLC_DLL_EXPORT friend std::ostream& operator << (std::ostream&, const badoublev&);

  ADOLC_DLL_EXPORT friend adubv operator+ (const badoublev &x);
  ADOLC_DLL_EXPORT friend adubv operator- (const badoublev &x);
  
  /* overload operations */
  ADOLC_DLL_EXPORT friend adubv operator+(const badoublev &op1,const badoublev &op2);
  ADOLC_DLL_EXPORT friend adubv operator-(const badoublev &op1,const badoublev &op2);
  ADOLC_DLL_EXPORT friend adubv operator*(const badoublev &op1, double n);
  ADOLC_DLL_EXPORT friend adubv operator*(double n, const badoublev &op1);

  //made inline by EGK
  ADOLC_DLL_EXPORT friend adub operator*(const badoublev &op1, const badoublev &op2);

  /* overloaded for interaction of constant and active vectors */
/* removed 1/95
  ADOLC_DLL_EXPORT friend adubv operator+(const badoublev &op1, const doublev &op2);
  ADOLC_DLL_EXPORT friend adubv operator+(const doublev &op1, const badoublev &op2);
*/
  ADOLC_DLL_EXPORT friend adubv operator+(const badoublev &op1, double* op2);
  ADOLC_DLL_EXPORT friend adubv operator+(double* op2, const badoublev &op1);
/* removed 1/95
  ADOLC_DLL_EXPORT friend adubv operator-(const badoublev &op1, const doublev &op2);
  ADOLC_DLL_EXPORT friend adubv operator-(const doublev &op1, const badoublev &op2);
*/
  ADOLC_DLL_EXPORT friend adubv operator-(const badoublev &op1, double* op2);
  ADOLC_DLL_EXPORT friend adubv operator-(double* op1, const badoublev &op2);
/* removed 1/95
  ADOLC_DLL_EXPORT friend adub operator*(const badoublev &op1, const doublev &op2);
  ADOLC_DLL_EXPORT friend adub operator*(const doublev &op1, const badoublev &op2);
*/
  ADOLC_DLL_EXPORT friend adub operator*(const badoublev &op1, double* op2);
  ADOLC_DLL_EXPORT friend adub operator*(double* op1, const badoublev &op2);
  
  /* overloaded for interaction of active scalars and active vectors */
/* removed 1/95
  ADOLC_DLL_EXPORT friend adubv operator/(const doublev &op1, const badouble &n);
*/
  ADOLC_DLL_EXPORT friend adubv operator*(const badoublev &op1, const badouble &n);
  ADOLC_DLL_EXPORT friend adubv operator*(const badouble &n, const badoublev &op1);
  /* excluded operations */
  err_retu operator>>=(double op1) {
/*     double x=op1;*/
     return("ADOL-C error: illegal argument combination for operator >>=\n");
  };
  err_retu operator<<=(double op1) {
/*     double x=op1;*/
     return("ADOL-C error: illegal argument combination for operator <<=\n");
  };
  err_retu operator+= (double op1) {
/*     double x=op1;*/
     return("ADOL-C error: illegal argument combination for operator +=\n");
  };
  err_retu operator-= (double op1) {
/*     double x=op1;*/
     return("ADOL-C error: illegal argument combination for operator -=\n");
  };
  inline ADOLC_DLL_EXPORT friend err_retu operator+(const badoublev &op1,double op2) {
/*     badoublev y=op1;
     double x=op2;*/
     return("ADOL-C error: illegal argument combination for operator +\n");
  };
  inline ADOLC_DLL_EXPORT friend err_retu operator-(const badoublev &op1,double op2) {
/*     badoublev y=op1;
     double x=op2;*/
     return("ADOL-C error: illegal argument combination for operator -\n");
  };
  inline ADOLC_DLL_EXPORT friend err_retu operator+(double op1,const badoublev &op2) {
/*     badoublev y=op2;
     double x=op1;*/
     return("ADOL-C error: illegal argument combination for operator +\n");
  };
  inline ADOLC_DLL_EXPORT friend err_retu operator-(double op1,const badoublev &op2) {
/*     badoublev y=op2;
     double x=op1;*/
     return("ADOL-C error: illegal argument combination for operator -\n");
  };
};


/****************************************************************************/
/*                                                              CLASS ADUBV */
class ADOLC_DLL_EXPORT adubv:public badoublev{
  adubv(int lo,int sz){start_loc=lo;size=sz;};
/* removed 1/95
  adubv(doublev&);
*/
  adubv():badoublev(0,0){
      std::cout << "ADOL-C error: illegal default construction of adub variable\n" ;
      exit(-2);
         };

 public:    

  //added by EGK
  ADOLC_DLL_EXPORT friend adub operator_mult ( const badoublev &arg1, const badoublev &arg2 );


/* removed 1/95
  ADOLC_DLL_EXPORT friend adub operator*(const badoublev &op1, const doublev &op2);
  ADOLC_DLL_EXPORT friend adub operator*(const doublev &op1, const badoublev &op2);
*/
  ADOLC_DLL_EXPORT friend adub operator*(const badoublev &op1, double* op2);
  ADOLC_DLL_EXPORT friend adub operator*(double* op1, const badoublev &op2);
  //made inline by EGK
  ADOLC_DLL_EXPORT friend adub operator*(const badoublev &op1, const badoublev &op2);
  /* excluded because g++ warnings
  ADOLC_DLL_EXPORT friend adub operator*(const badoublev &op1, const doublev &op2);
  ADOLC_DLL_EXPORT friend adub operator*(const doublev &op1, const badoublev &op2);
  */
/* removed 1/95
  ADOLC_DLL_EXPORT friend adubv operator+(const badoublev &op1, const doublev &op2);
  ADOLC_DLL_EXPORT friend adubv operator+(const doublev &op1, const badoublev &op2);
  ADOLC_DLL_EXPORT friend adubv operator-(const badoublev &op1, const doublev &op2);
  ADOLC_DLL_EXPORT friend adubv operator-(const doublev &op1, const badoublev &op2);
  ADOLC_DLL_EXPORT friend adubv operator/(const doublev &op1, const badouble &n);
  ADOLC_DLL_EXPORT friend adubv operator*(const doublev &op1, const badouble &n);
  ADOLC_DLL_EXPORT friend adubv operator*(const badouble &n, const doublev &op1);
*/
  ADOLC_DLL_EXPORT friend adubv operator/(const badoublev &op1, const badouble &n);
  inline ADOLC_DLL_EXPORT friend adubv operator/(const badoublev &op1, double n);
  ADOLC_DLL_EXPORT friend adubv operator+ (const badoublev &x);
  ADOLC_DLL_EXPORT friend adubv operator- (const badoublev &x);
  ADOLC_DLL_EXPORT friend adubv operator+(const badoublev &op1,const badoublev &op2);
  ADOLC_DLL_EXPORT friend adubv operator-(const badoublev &op1,const badoublev &op2);
  ADOLC_DLL_EXPORT friend adubv operator*(const badoublev &op1, double n);
  ADOLC_DLL_EXPORT friend adubv operator*(double n, const badoublev &op1);
  /* excluded because g++ warnings
  ADOLC_DLL_EXPORT friend adubv operator+(const badoublev &op1, const doublev &op2);
  ADOLC_DLL_EXPORT friend adubv operator+(const doublev &op1, const badoublev &op2);
  */
  ADOLC_DLL_EXPORT friend adubv operator+(const badoublev &op1, double* op2);
  ADOLC_DLL_EXPORT friend adubv operator+(double* op2, const badoublev &op1);
  /* excluded because g++ warnings
  ADOLC_DLL_EXPORT friend adubv operator-(const badoublev &op1, const doublev &op2);
  ADOLC_DLL_EXPORT friend adubv operator-(const doublev &op1, const badoublev &op2);
  */
  ADOLC_DLL_EXPORT friend adubv operator-(const badoublev &op1, double* op2);
  ADOLC_DLL_EXPORT friend adubv operator-(double* op1, const badoublev &op2);
  /* excluded because g++ warnings
  ADOLC_DLL_EXPORT friend adubv operator/(const doublev &op1, const badouble &n);
  */
  ADOLC_DLL_EXPORT friend adubv operator*(const badoublev &op1, const badouble &n);
  ADOLC_DLL_EXPORT friend adubv operator*(const badouble &n, const badoublev &op1);
#ifdef overwrite
  ~adubv();
#endif
};


/****************************************************************************/
/*                                                           CLASS ADOUBLEV */
class ADOLC_DLL_EXPORT adoublev:public badoublev
{
  ADOLC_DLL_EXPORT friend class adoublem;
  adoublev(){};
 public:
  adoublev(const adubv& a);
  adoublev(const adoublev&);
  adoublev(int sz);
//  adoublev(int n, double *values);
/* removed 1/95
  adoublev(doublev&);
*/

#ifdef overwrite
  ~adoublev();
#endif
/* removed 1/95
  adoublev& operator= (const doublev &y);
*/
  adoublev& operator= (const badoublev&);
  adoublev& operator= (const adoublev&);
  adoublev& operator= (const adubv&);
  adoublev& operator= (double y);
  adoublev& operator= (double* y);

      // These two functions are meant to replace the previous function so
      // that less copying takes place. First call operator_equal_start()
      // which returns a pointer to the begining of the buffer which you were
      // planning on setting equal to double* y. Then set the buffer to the
      // values. Then call operator_equal_end ().
      double* operator_equal_start ();
      void operator_equal_end ();

/* removed 1/95
  adoublev& operator >>= (doublev& );
  adoublev& operator <<= (doublev& );
*/
  adoublev& operator >>= (double* );
  adoublev& operator <<= (double* );
};

/*--------------------------------------------------------------------------*/
inline adubv operator / (const badoublev& x, double y){return (1.0/y)*x;}


/****************************************************************************/
/*                                                           CLASS ADOUBLEM */
class ADOLC_DLL_EXPORT adoublem
{
  int n, m;          /* Size of the matrix */
  adoublev *index;     /* So each row is an adoublev */
 public:
  adoublem(int n, int m);
  adoublem(const adoublem& );
  ~adoublem();

// EGK: made inline by EGK, removed bounds checking
   adoublev& operator[](int i)  /* Can access component like an array */
      {
#ifdef BOUNDS_CHECK
	 if (i<0 || i>=m)
	 { fprintf (DIAG_OUT,"ADOL-C error: adoublem index out of range.\n");
	 exit(-3);
	 }
#endif
	 return index[i];}

// EGK: added by EGK
   const adoublev& operator[](int i) const /*Can access component like an array*/
      {
#ifdef BOUNDS_CHECK
	 if (i<0 || i>=m)
	 { fprintf (DIAG_OUT,"ADOL-C error: adoublem index out of range.\n");
	 exit(-3);
	 }
#endif
	 return index[i];}

   asubv operator[](const along&);
};


/****************************************************************************/
/*                                                              CLASS ASUBV */
class ADOLC_DLL_EXPORT asubv:public badoublev
{
  locint base,offset,begin;
  public:
  asubv(adoublev* start, locint index);
#ifdef overwrite
    ~asubv();
#endif
/* removed 1/95
  asubv& operator <<= (doublev&);
  asubv& operator = (doublev);
*/
  asubv& operator <<= (double*);
  asubv& operator = (double*);
  asubv& operator = (const adubv&);
  asubv& operator = (const badoublev&);
/* added Sep/01/96 */
  asubv& operator = (const asubv&);
  asubv& operator += (const badoublev&);
  asubv& operator -= (const badoublev&);
/* there are currently no +=, -= operators for double* 
   right hand sides. They woudl require a special treatment 
   similar to the assignment operators caused by the buffered
   writing of the constant right hand side to the tape. */
  asubv& operator *= (double x);
  asubv& operator *= (const badouble&);
  asubv& operator /= (double x);
  asubv& operator /= (const badouble&);
};




// These were made inline by EGK
/*
inline adub operator* ( const badoublev &arg1, const badoublev &arg2 ) 
{ 

  if (trace_flag) 
  { 
     return operator_mult(arg1,arg2);
  }

  

  double x = 0;    

//  for (int i=0; i<arg1.size; i++)
//     x += store[arg1.start_loc+i] * store[arg2.start_loc+i];


  const double* p1 = &store[arg1.start_loc];
  const double* p2 = &store[arg2.start_loc];
  const int arg1size = arg1.size;
  for (int i=0; i<arg1size; i++)
     x += (*p1++) * (*p2++);

  const locint locat = next_loc_fast();

  store[locat] = x;
  return locat;
}


inline adoublev& adoublev::operator= ( double* coval ) 
{ if (trace_flag) 
  { 
     return operator_equal(coval);
  }

  for (int i=0; i<size; i++)
    store[start_loc+i] = coval[i];
  return *this;
}
*/



/****************************************************************************/
/*                                                                THAT'S ALL*/

#endif /* __cplusplus */
#endif /* ADOLC_TAPELSS */
#endif /* ADOLC_AVECTOR_H */


//@@@@@@ The following is not a copy of any adolc file. We simply add here
// some more function prototypes.

int revolve_fos_reverse(short   tnum,
			int     depen,    
			int     indep,    
			double  *lagrange,
			double  *results); 
int fos_reverse(short   tnum,
		int     depen,    
		int     indep,    
		double  *lagrange,
		double  *results); 
double* get_adjoints_ptr();
revreal* get_T_ptr();
extern int currently_in_initial_timestep;

extern int trace_flag;
extern locint location_cnt;

