##' Splitting variables
##' 
##' This function extracts the information of split variables from tree output.
##' 
##' @param tree.result The tree output from cctree.
##' @return A vector of characteristics giving the name of each splitting variable in the tree.
##' @author Jianchang Hu
##' @examples 
##' set.seed(12345 + 987)
##' n_sample <- 2000
##' n_var <- 200
##' maf <- runif(n_var, 0.05, 0.45)
##' x <- sapply(maf, function(snp.maf){sample(x=c(2,1,0), n_sample, replace=TRUE, 
##' prob=c(snp.maf^2, 2* snp.maf *(1-snp.maf), (1-snp.maf)^2))})
##' colnames(x) <- paste0("X_", seq(n_var))
##' 
##' log.odds <- -1 + (x[, 2] > 0)
##' prob.case <- exp(log.odds) / (1 + exp(log.odds))
##' Y <- sapply(prob.case, rbinom, n = 1, size = 1)
##' 
##' data <- cbind(Y, x)
##' test.tree <- tree_build(data, method = "entropy") # build the tree for a data set
##' split.var.name <- split_var_extract(test.tree)
##' @export
split_var_extract = function(tree.result){
      delete=which(colnames(tree.result$learning.data) == "affected")
      all_vars=colnames(tree.result$learning.data)[-delete] # all covariates and SNPs considered
      
      split_vars=NULL
      for ( i in 1:tree.result$nnd ) {
            if ( is.na(tree.result$spv[i]) ) next() # if there NA at current node, then move on
            val0=strsplit(tree.result$spv[i],"x")
            val=as.numeric(val0[[1]][2])
            split_vars=c(split_vars, all_vars[val]) # add the SNP used here to the list
      }
      return(split_vars) # output the final list
}
