##' Depth of Nodes
##' 
##' This function learns the depth of the node based on the "pt" information from tree output.
##' 
##' @param tree.result The tree output from cctree.
##' @return A vector of integers giving the depth of each node in the tree.
##' @author Jianchang Hu
##' @examples 
##' set.seed(12345 + 987)
##' n_sample <- 2000
##' n_var <- 200
##' maf <- runif(n_var, 0.05, 0.45)
##' x <- sapply(maf, function(snp.maf){sample(x=c(2,1,0), n_sample, replace=TRUE, 
##' prob=c(snp.maf^2, 2* snp.maf *(1-snp.maf), (1-snp.maf)^2))})
##' colnames(x) <- paste0("X_", seq(n_var))
##' 
##' log.odds <- -1 + (x[, 2] > 0)
##' prob.case <- exp(log.odds) / (1 + exp(log.odds))
##' Y <- sapply(prob.case, rbinom, n = 1, size = 1)
##' 
##' data <- cbind(Y, x)
##' test.tree <- tree_build(data, method = "entropy") # build the tree for a data set
##' test.depth <- depth_find(test.tree)
##' @export
depth_find <- function(tree.result){
      depth.node <- function(current.node){
            if(current.node == 1) return(1) # root node
            
            depth <- 2 # other node has depth at least 2
            nd.parent <- tree.result$pt[current.node]
            while(nd.parent != 0){
                  nd.parent <- tree.result$pt[nd.parent + 1]
                  depth <- depth + 1
            }
            # check if the parent node is root or not
            # if its not, go up one layer and depth add 1
            return(depth)
      }
      
      num.nodes <- tree.result$nnd # number of nodes in the tree
      sapply(seq(num.nodes), depth.node) # depth for all nodes in the tree
}
